/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 � i-net software 1998-2013

*/
using System.Collections.Generic;
using System;
using Inet.Viewer.Data;

namespace Inet.Viewer
{
    /// <summary>
    /// This enum defines the different available search options. This enum is build so the different
    /// options can be used at the same time.
    /// <seealso cref= "IRenderData.Search(String, int, SearchOption)"> </seealso>
    /// </summary>
    public enum SearchOption
    {
        /// <summary>
        /// By default there are no search options
        /// </summary>
        None = 0,

        /// <summary>
        /// searches the word in its entirety, not parts of the word</summary>
        WholeWord = 1,

        /// <summary>
        /// Distinguishes upper and lower cases in the search. E.g. "Search" != "search" </summary>
        CaseSensitive = 2,

        /// <summary>       
        /// The search phrase is to be handled as a regular expression (using Java's Regex methods for the search) </summary>
        RegularExpression = 4
    }
 
    /// <summary>
    /// This is the interface used by ReportViews to fetch the report data.
    /// In order to create a working ReportView and add it to the ReportViewer, an instance of
    /// ReportData must be given, which will then be used to fetch the report's render data.
    /// 
    /// <p><h3>Notes for implementing</h3></p>
    /// <p><ul><li>Implementations of this interface must be thread safe except for the methods 
    /// <code>getExportChunkCount</code> and <code>getNextExportChunk</code>.</li>
    /// <li>Note that each ReportView can have its own instance of ReportData, so the ReportData methods
    /// may be called simultaneously for different ReportViews and different ReportData instances.</li>
    /// <li>Note that getCopy() should be implemented if extending from another ReportData, since
    /// it otherwise will create an instance of the superclass rather than your own ReportData.</li></ul></p>  
    /// </summary>   
    /// <remarks> Implements ICloneable:
    /// "Clones" this ReportData object with all its properties and returns the copy. Useful for deriving from existing
    /// ReportData objects by copying them and adding or changing properties. This method is called by the viewer for drilling down,
    /// for example - the drilldown property is set on the copy while all other properties remain the same, and the copy is used to
    /// open a new report view. </remarks>      
    public interface IRenderData : ICloneable
    {

        /// <summary>
        /// Sets the title of the report. This title is what would be typically displayed in a title bar, e.g. "Employee Report 2005".
        /// Note that if this is not manually set, the title will be whatever the report has set as its title.
        /// If a title is set, it overrides the "actual" report title. </summary>
        /// <param> Title of the report as simple string.</param>
        string ReportTitle { set; get; }


        /// <summary>
        /// Sets the location of the report which this ReportData is to obtain its data for. 
        /// Any properties can be attached to the end using "=" for separating property name and value, and "&amp;" for
        /// separating the various properties, e.g. "...&amp;init=pdf&amp;promptABC=test..." The location may not be null. </summary>
        /// <param> The location at which the report can be found. May not be null. *</param>
        string ReportLocation { set; get; }

        /// <summary>
        /// Returns the binary data of one page for the .NET Viewer. If refresh is false a cached version will be returned. with refresh set to true 
        /// the report will be enforced to be rerendered.
        /// </summary>
        /// <param name="page">The page number. The first page is 1, the second is 2, ... .
        /// If the page number is bigger than the page count the data from the last page will be returned.
        /// </param>
        /// <param name= "refresh">If set to true enforces the whole report to be re-rendered on server side and returns the refreshed binary data. This 
        /// page is rendered anew rather than possibly taking a cached version.     </param>   
        ///  /// <returns> The binary data of the page for the .NET Viewer, or possibly null if page number is not valid. </returns>
        /// <seealso cref= "GetPageCount()"/>
        /// <exception cref="ViewerException">  If an exception is returned from the server, it is wrapped in a ViewerException.</exception>
        byte[] GetPageData(int page, bool refresh);

        /// <summary>
        /// Returns the number of pages in the report. <BR/>
        /// This method blocks until the rendering process is finished. This is useful if you use i-net Clear Reports with
        /// external result sets or connections and you want to know when the rendering process is finished and you can close
        /// these external result sets or connections - once this method returns, the rendering process is finished.
        /// Note that this method is thread-safe, that is, two or more threads concurrently calling
        /// this method and the other thread-safe methods in ReportData should cause no problems.
        /// </summary>
        /// <returns> number of pages (1-based) </returns>
        /// <exception cref= "ViewerException"> If there are rendering problems, etc. </exception>
        /// <seealso cref= "GetPageData(int, bool)"></seealso>
        int GetPageCount();

        /// <summary>
        /// Check if the rendering of the report ran into a page limit. This means does not all possible pages exist.
        /// </summary>
        /// <returns> true, if there is a limit </returns>
        /// <exception cref= "ViewerException">If there are rendering problems, etc.</exception>
        bool IsPageLimitExceeded { get; }

        /// <summary>
        /// Returns the next chunk of the exported report if there is at least one more chunk available. 
        /// It is required that getExportChunkCount(Properties) has been called once for this report.<BR/>
        /// This method blocks until the rendering process of the requested chunk or the complete report is finished. </summary>
        /// <returns> Next chunk of the report, or null if the export is finished. </returns>
        /// <seealso cref= "GetExportChunkCount(Dictionary{string,string})"> </seealso>
        /// <exception cref="ViewerException"> If there are connection problems or other issues while fetching the data</exception>        
        byte[] NextExportChunk();


        /// <summary>
        /// Returns the total number of export "chunks" for the report for the specified export format. <BR/>
        /// A "chunk" is a unit of export data which can be retrieved using getNextExportChunk(). 
        /// This method blocks until the rendering process is finished on the server. Note that this method is
        /// thread-safe, that is, two or more threads concurrently calling this method and the other 
        /// thread-safe methods in ReportData should cause no problems.<BR/>
        /// In case of gzip compression, the number of chunks can not be determined and "0" is returned.
        /// The chunks can be retrieved in a loop until GetNextExportChunk() returns NULL.
        /// </summary>
        /// <param name="exportParameter"> Parameter to be appended to the current report Parameter for this export. 
        /// The Parameter should contain at least the export format with the key "export_fmt". The following
        ///            formats are valid:<Br/>
        ///            "pdf" - pdf file<Br/>
        ///            "rtf" - rtf file<Br/>
        ///            "xls" - xls file<Br/>
        ///            "csv" - csv file<Br/>
        ///            "ps" - ps file<Br/>
        ///            "xml" -xml file<Br/>
        ///            "htm" - html files, The file name is the base filename, because there are more files with links.<Br/>
        /// For a complete list of report URL properties see the documentation.
        /// </param>
        /// <returns> Total number of export "chunks" (i.e. units) which can be fetched using getNextExportChunk() </returns>
        /// <seealso cref = "NextExportChunk()"></seealso>
        /// <seealso cref = "this[string]"> </seealso>
        /// <exception cref = "ViewerException"> If there are connection problems or other issues while fetching the data </exception>      
        int GetExportChunkCount(Dictionary<string, string> exportParameter);

        /// <summary>
        /// Returns the group tree which can be sent to the java viewer as a byte array. 
        /// Note that this method is thread-safe, that is, two or more threads concurrently calling this method and the other
        /// thread-safe methods in ReportData should cause no problems. </summary>
        /// <returns> byte[] Group Tree as byte array, encoded by i-net Clear Reports. </returns>
        /// <exception cref="ViewerException"> If there are connection problems or other issues while fetching the data</exception>  
        byte[] GetGrouptreeData();


        /// <summary>
        /// Set or get the chosen URL parameter value which has been set either via setReportProperty or otherwise.
        /// See <a href="https://docs.inetsoftware.de/reporting/help/report-url-parameters" target="_blank">report url parameters</a>
        /// for a list of possible properties to set here and what they mean.<br/>
        /// Setting null as the value will cause the removal of the property. If a prompt property is
        /// to be explicitly set to the value "null", simply set the string "formula:null" as the value. 
        /// </summary>
        /// <param name="key"> Name of property to get value for </param>
        /// <returns> Value of property with the given name </returns>
        string this[string key] { get; set; }

        /// <summary>
        /// Sets the property "promptOnRefresh", that is, whether prompts are to be shown whenever the report is refreshed, that is,
        /// whenever new data is fetched from the server. </summary>
        /// <param> Value to set for this property</param>
        bool PromptOnRefresh { set; get; }

        /// <summary>
        /// Stops and cancels the rendering process if one is running. This command ends up
        /// being passed through to i-net Clear Reports rendering engine and running Engine.stop.
        /// </summary>
        void Stop();

        /// <summary>
        /// Searches the given phrase in the report, starting at a certain page and using
        /// the search options given in the <see cref="SearchOption">flags</see>. The flags
        /// be combined with the  OR operator "|". So, for example, to search with the flags WholeWord and CaseSensitive, use
        /// "SearchOption.WholeWord | SearchOption.CaseSensitive" as option.<p/>
        /// 
        /// Note that phrases going over more than one page will not be found, nor will text
        /// parts with formatting inside the word itself, such as t<b>hi</b>s word. 
        /// Note that this method is thread-safe, that is, two or more threads concurrently calling
        /// this method and the other thread-safe methods in ReportData should cause no problems.
        /// </summary>
        /// <param name="phrase"> Word or phrase to search. Should be a regular expression if this flag is set. </param>
        /// <param name="startPage"> Page to start searching on (1-based) </param>
        /// <param name="flags"> Desired search options <see cref="SearchOption"/></param>
        /// <returns> Search result by the server, encoded in the i-net Clear Reports protocol. </returns>
        byte[] Search(string phrase, int startPage, SearchOption flags);


        /// <summary>
        /// Returns the byte array of the embedded font at the specified fontId,
        /// which is encoded in the Viewer's protocol. You need not and should not call this method yourself - rather the Viewer
        /// will call this method on its own when an embedded font needs to be fetched for the report.<p/>
        /// The fontID is 1-based. <code>null</code> will be returned if there are no fonts embedded for the
        /// report of this Engine. If fontID is greater than the number of embedded
        /// fonts available it will return the last font available. This method is the mirror method to com.inet.report.Engine.getFontData(int),
        /// so implementations can either call that if an engine is available, or a running i-net Clear Reports server
        /// also returns the font data with an HTTP request of the form
        /// "http://server:9000/?report=report.rpt&amp;export_fmt=font&amp;page=FONTID&amp;cmd=get_pg" where FONTID is replaced with the given font ID.
        /// Note that this method is thread-safe, that is, two or more threads concurrently calling
        /// this method and the other thread-safe methods in ReportData should cause no problems. </summary>
        /// <param name="fontID"> ID of font as specified in the i-net Clear Reports protocol </param>
        /// <returns> Partial or whole font with the ID, and with the most current version of this font known to the server.</returns> 
        byte[] GetFontData(int fontID);

        /// <summary>
        /// Notifies the server that the report with the currently set properties is
        /// still being viewed and therefore should not be removed from the cache yet
        /// if there is one. This method does NOT cause the report to be re-rendered under
        /// any circumstance, rather it resets the cache timeout on the server if there
        /// is one.<p>
        /// This method will be called in regular intervals (default is every 5 minutes)
        /// by its corresponding ReportView(s).</p> </summary>
        /// <exception cref="ViewerException"> if resetting the cache timeout is not supported by the ReportData
        /// implementation. This will cause the viewer to stop calling this method in regular intervals. </exception>
        /// <returns>The state of this report on the server, or null if this is not supported (requires version 13.1)</returns> 
        ReportState Ping();
    }
}